package org.yenc;
import java.io.*;

/**
 * Implementation of the <b>decoder</b> for YEncoding project.
 * <P>
 * This class is to be used to decode files encoded with yenc<br>
 * <FONT Size=+2>See <a href="http://www.yenc.org">www.yenc.org</a> for details.</FONT>
 * <P>
 * To <b>run</b> the project, use:<br>
 * <code>java org.yenc.YDecoder FileToDecode DestinationFolder</code>
 * <P>
 * Known limitations:
 * <UL>
 *     * The CRC is not checked.<br>
 *     * I tried using Streams, but those do not support binary very well.<br>
 * </UL>
 * <P>
 *
 * If you have imporovements to this code, please send them to me or to yenc@infostar.de
 * <P>
 * @author &lt; Alex Rass &gt; sashasemail@yahoo.com
 * @version 2<br>
 * Copywrite by Alex Rass 2002.
 * This software is to be distributed in accordance with the GNU piblic license.
 */
final public class YDecoder {
    private static final String EMPTY_STRING="";
    static {
        System.err.println("Decoder for YEnc.org project.  Version "+ getVersionNumber());
    }

    /**
     * Making this private, ensures that noone tries to instantiate this class.
     */
    private YDecoder () {
    }

    /**
     * This method does all of the decoding work.
     *
     * @param file   takes a file to read from
     * @param folder destination folder.
     *               File will be created based on the name provided by the header.
     *
     *               if there is an error in the header and the name
     *               can not be obtained, "unknown" is used.
     * @exception IOException
     */
    public static void decode(RandomAccessFile file, String folder) throws IOException{

      /* Get initial parameters */
      String line = file.readLine();
      while (line!=null && !line.startsWith("=ybegin")) {
        line = file.readLine();
      }
      if (line==null)
        return;

      String fileName = parseForName(line, "name");
      if (fileName==null)
          fileName = "Unknown.blob";
      fileName = folder + fileName;
      RandomAccessFile fileOut = new RandomAccessFile(fileName, "rw");

      String partNo = parseForName(line, "part");

      /* Handle Multi-part */
      if (partNo!=null) {
          while (line!=null && !line.startsWith("=ypart")) {
            line = file.readLine();
          }
          if (line==null)
            return;

          /* Get part-related parameters */
          long begin = Long.parseLong(parseForName(line, "begin")) - 1;
          if (fileOut.length()<begin)
            fileOut.setLength(begin-1); // reset file
          fileOut.seek(begin);
      } else {
          fileOut.setLength(0); // reset file
      }

      /* Decode the file */
      int character;
      boolean special=false;

      line = file.readLine();
      while (line!=null && !line.startsWith("=yend")) {
          for (int lcv=0;lcv<line.length(); lcv++){
            character = (int)line.charAt(lcv);
            if (character != 61) {
                character = decodeChar(character, special);
                fileOut.write(character);
                //System.out.print((char) character);
                special = false;
            } else
                special = true;
          }
          line = file.readLine();
      }
      fileOut.close();
    }

    private static int decodeChar(int character, boolean special) throws IOException {
        int result;
        if (special)
          character = character-64;

        result = character-42;

        if (result<0)
          result += 256;

        return result;
    }

    private static String parseForName (String line, String param) {
        int indexStart = line.indexOf(param+"=");
        int indexEnd = line.indexOf(" ", indexStart);
        if (indexEnd==-1)
          indexEnd = line.length() ;
        if (indexStart>-1)
          return line.substring(indexStart+param.length()+1, indexEnd);
        else
          return null;
    }

    /**
     * Provides a way to find out which version this decoding engine is up to.
     *
     * @return Version number
     */
    public static int getVersionNumber(){
        return 2;
    }

    /**
     * To Run:
     *  java org.yenc.YDecImpl FileToDecode DestinationFolder
     *
     * @param args   Command line argument(s)
     * @exception IOException
     */
    public static void main (String args[]) throws IOException{
        if (args.length < 1) {
            // print usage and exit
            System.err.println("Usage arguments: fileIn folderOut");
        } else {
            RandomAccessFile file = new RandomAccessFile(args[0],"r");
            String folder = args.length>1?args[1]+File.separator:"";
            decode(file, folder);
        }
    }
}
